#ifndef SCARD_REG_H
#define SCARD_REG_H

#include "reader/support.h"

#define INVALID_STR ((size_t)-1)
#define INVALID_CHAR ((unsigned char)-1)

static SUP_INLINE unsigned char c2d(TCHAR c)
{
    if (c >= '0' && c <= '9') return (unsigned char)(c - '0');
    if (c >= 'A' && c <= 'F') return (unsigned char)(c - 'A' + 10);
    if (c >= 'a' && c <= 'f') return (unsigned char)(c - 'a' + 10);
    return INVALID_CHAR;
}

static SUP_INLINE size_t str2bin(const TCHAR *str, unsigned char *data)
{
    size_t length = 0;
    for (; *str; str += 2, data++, length++)
    {
	unsigned char hi, lo;
	if (!str[1]) return INVALID_STR;
	hi = c2d(*str);
	lo = c2d(str[1]);
	if (hi == INVALID_CHAR || lo == INVALID_CHAR)
	    return INVALID_STR;
	*data = (hi << 4) | lo;
    }
    return length;
}

static SUP_INLINE TCHAR d2c(unsigned char d)
{
    if (d >= 10) return d - 10 + 'A';
    return d + '0';
}

static  SUP_INLINE void bin2str(const unsigned char *data, size_t length,
    TCHAR *str)
{
    for (; length; length--, str += 2, data++)
    {
	*str = d2c(*data >> 4);
	str[1] = d2c(*data & 0xF);
    }
    *str = 0;
}
static const TCHAR BASE_PATH[] = _TEXT("\\config\\KeyCarriers\\");
static const TCHAR ATR[] = _TEXT("ATR");
static const TCHAR MASK[] = _TEXT("Mask");
static const TCHAR NAME[] = _TEXT("Name");
static const TCHAR FOLDERS[] = _TEXT("Folders");
static const TCHAR FILTER[] = _TEXT("Filter");
static const TCHAR * default_connect = _TEXT("Default");

static SUP_INLINE DWORD register_carrier_internal(const TCHAR* carrier, const TCHAR* connect, const BYTE* atr, size_t atr_len, const BYTE* mask, size_t mask_len, const TCHAR* name, const TCHAR* folders, const BYTE* filter, size_t filter_len) {
    const TCHAR* cfg = connect
	? connect
	: default_connect;
    const TCHAR* format = _TEXT("%s\\%s\\%s\\%s");
    size_t path_length = 3 //    format
	+ _tcslen(BASE_PATH)
	+ _tcslen(carrier)
	+ _tcslen(cfg)
	+ _tcslen(FOLDERS) //        : ATR, Mask, Name, Folders, Filter
	+ 1; // null-

    TCHAR *path = (TCHAR*)malloc(path_length * sizeof(TCHAR));
    if (!path)
	return (DWORD)NTE_NO_MEMORY;

    DWORD code = ERROR_INTERNAL_ERROR;

    _sntprintf(path, path_length, format, BASE_PATH, carrier, cfg, ATR);
    code = support_registry_put_hex(path, atr_len, atr);
    if (code != ERROR_SUCCESS) {
	goto done;
    }

    _sntprintf(path, path_length, format, BASE_PATH, carrier, cfg, MASK);
    code = support_registry_put_hex(path, mask_len, mask);
    if (code != ERROR_SUCCESS) {
	goto done;
    }

    if (name) {
	_sntprintf(path, path_length, format, BASE_PATH, carrier, cfg, NAME);
	code = support_registry_put_string(path, name);
	if (code != ERROR_SUCCESS) {
	    goto done;
	}
    }
    if (folders) {
	_sntprintf(path, path_length, format, BASE_PATH, carrier, cfg, FOLDERS);
	code = support_registry_put_string(path, folders);
	if (code != ERROR_SUCCESS) {
	    goto done;
	}
    }
    if (filter) {
	_sntprintf(path, path_length, format, BASE_PATH, carrier, cfg, FILTER);
	code = support_registry_put_hex(path, filter_len, filter);
	if (code != ERROR_SUCCESS) {
	    goto done;
	}
    }
    code = ERROR_SUCCESS;

done:
    free(path);

    return code;
}

static SUP_INLINE DWORD	register_carrier_ex(const TCHAR *carrier, const TCHAR *connect, const TCHAR *atr, const TCHAR *mask, const TCHAR *name, const TCHAR *folder, const TCHAR *filter) {
    BYTE *atr_bytes = NULL;
    BYTE *mask_bytes = NULL;
    BYTE *filter_bytes = NULL;
    size_t atr_len_in_bytes = _tcslen(atr) / 2;
    size_t mask_len_in_bytes = _tcslen(mask) / 2;
    size_t filter_len_in_bytes = filter ? (_tcslen(filter) / 2) : 0;
    DWORD err = 0;

    if (atr_len_in_bytes != mask_len_in_bytes)
	return (DWORD)ERROR_INVALID_DATA;

    atr_bytes = (BYTE *)malloc(atr_len_in_bytes);
    if (!atr_bytes)
	return (DWORD)NTE_NO_MEMORY;
    mask_bytes = (BYTE *)malloc(mask_len_in_bytes);
    if (!mask_bytes) {
	err = (DWORD)NTE_NO_MEMORY;
	goto done;
    }
    if (filter_len_in_bytes)
    {
	filter_bytes = (BYTE *)malloc(filter_len_in_bytes);
	if (!filter_bytes) {
	    err = (DWORD)NTE_NO_MEMORY;
	    goto done;
	}
    }
    if (INVALID_STR == str2bin(atr, atr_bytes))
    {
	err = (DWORD)ERROR_INVALID_DATA;
	goto done;
    }
    if (INVALID_STR == str2bin(mask, mask_bytes))
    {
	err = (DWORD)ERROR_INVALID_DATA;
	goto done;
    }
    if (filter_len_in_bytes && (INVALID_STR == str2bin(filter, filter_bytes)))
    {
	err = (DWORD)ERROR_INVALID_DATA;
	goto done;
    }
    err = register_carrier_internal(carrier, connect, atr_bytes, atr_len_in_bytes, mask_bytes, mask_len_in_bytes, name, folder, filter_bytes, filter_len_in_bytes);

done:
    free(atr_bytes);
    free(mask_bytes);
    free(filter_bytes);

    return err;
}

static SUP_INLINE DWORD	register_carrier(const TCHAR* carrier, const TCHAR* connect, const TCHAR* atr, const TCHAR* mask, const TCHAR* name, const TCHAR* folder) {
    return register_carrier_ex(carrier, connect, atr, mask, name, folder, NULL);
}
 
#endif //SCARD_REG_H
