/*
 * Copyright(C) 2000  
 *
 *    , 
 *    .
 *
 *        ,
 * ,    ,
 *     ,
 * ,      
 *     
 *      .
 */

/****
 * $Id: dirall.c 283063 2024-12-10 06:55:44Z sdd $
 *
 *       (READER).
 *
 *   /      (FAT12).
 *
 *  ,      ,   .
 *
 * : FAT12_INFO_SUPPORT
 ****/

#include "fat12prj.h"
#include "cpcsp/visibility.h"
#include "reader.kit/rdr_ver.h"
#include "os_specific.h"

static const TCHAR BASE_NAME_CFG[] = _TEXT("DIR");
static const TCHAR DEFAULT_GROUP_CONNECT_CFG[] = _TEXT("PNP DIR");

static void free_FOLDER_ITEM(FOLDER_ITEM * items, DWORD items_number, TCHAR * full_buffer)
{
#ifdef _WIN32
    DWORD i;
    if (items) {
	for (i = 0; i < items_number; i++)
	    if (items[i].path != NULL)
		free(items[i].path);
    }
#else
    UNUSED(items_number);
#endif /*_WIN32*/
    free(items);
    free(full_buffer);
}

static SUP_INLINE DWORD get_check_mode_from_registry()
{
    static const TCHAR ItemsPath[] = _TEXT("\\config\\KeyDevices\\DIR\\PNP DIR\\Default\\CheckMode");
    long tmp_res = 0;
    DWORD err = support_registry_get_long(ItemsPath, &tmp_res);
    if (err)
	return 0;
    return tmp_res != 0 ? 1 : 0;
}

/*!
 * \ingroup fat12_info_general
 * \brief   .
 * \return  
 * \retval ERROR_SUCCESS  
 * \retval (DWORD)NTE_NO_MEMORY   
 */
DWORD dir_register(TSupSysContext *context, TSupSysInfo *info)
{
    TFat12Context *ctx;
    TSupSysInfoRegister *inf = (TSupSysInfoRegister*)info;
    TCHAR * reg_path = NULL;
    size_t reg_length;
    DWORD err = ERROR_SUCCESS;
    FOLDER_ITEM * items = NULL;
    DWORD items_number = 0, i = 0;
    TCHAR * full_buffer = NULL;
    size_t max_name_length = 0;
    size_t str_last_pos;
    BOOL found = FALSE;
    UNUSED(context);

    SUPSYS_PRE_INFO(info, TSupSysInfoRegister);
    SUPSYS_PRE_STRING_PTR(inf->in_group_name.text);

    support_set_library(inf->support_module);

    ctx = calloc(1, sizeof(TFat12Context));
    if (ctx == NULL) {
	return (DWORD)NTE_NO_MEMORY;
    }
    _tcsncpy(ctx->nickname, inf->in_group_name.text, sizeof(ctx->nickname) / sizeof(TCHAR) - 1);
    ctx->nickname[SUPSYSE_NICKNAME_LENGTH] = 0;
    _tcsncpy(ctx->connect, inf->in_group_name.text, sizeof(ctx->connect) / sizeof(TCHAR) - 1);
    ctx->connect[SUPSYSE_NICKNAME_LENGTH] = 0;
    ctx->must_check_connect = get_check_mode_from_registry() == 1;
    err = get_items_from_registry(NULL, NULL, &items, &items_number, &full_buffer, &max_name_length);
    if (err)
	goto done;
    if (!_tcscmp(inf->in_group_name.text, DEFAULT_GROUP_CONNECT_CFG)) {
	if (items_number > 0) {
	    i = 0;
	    found = TRUE;
	}
    }
    else {
	for (i = 0; i < items_number; i++) {
	    if (!_tcscmp(items[i].name, ctx->connect)) {
		found = TRUE;
		break;
	    }
	}
    }
    if (!found) {
	err = (DWORD)SCARD_E_READER_UNAVAILABLE;
	goto done;
    }
    ctx->path = malloc(sizeof(TCHAR)*(_tcslen(items[i].path) + 2));
    if (!ctx->path) {
	err = (DWORD)NTE_NO_MEMORY;
	goto done;
    }
    _tcscpy(ctx->path, items[i].path);
    str_last_pos = _tcslen(ctx->path) - 1;
    if (ctx->path[str_last_pos] != _TEXT('\\') && ctx->path[str_last_pos] != _TEXT('/'))
    {
#ifdef UNIX
	_tcscat(ctx->path, _TEXT("/"));
#else
	_tcscat(ctx->path, _TEXT("\\"));
#endif/*UNIX*/
    }
    ctx->volume_label = malloc(sizeof(TCHAR)*(_tcslen(items[i].unique) + 1));
    if (!ctx->volume_label) {
	err = (DWORD)NTE_NO_MEMORY;
	goto done;
    }
    _tcscpy(ctx->volume_label, items[i].unique);
    reg_length = _tcslen(BASE_PATH_CFG) + 1 + _tcslen(BASE_NAME_CFG) + 1 + _tcslen(inf->in_group_name.text) + 1 + _tcslen(DEFAULT_CONNECT) + 1 + _tcslen(LONG_NAMES_KEY) + 1;
    reg_path = malloc(reg_length * sizeof(TCHAR));
    if (!reg_path) {
	err = (DWORD)NTE_NO_MEMORY;
	goto done;
    }
    _sntprintf(reg_path, reg_length, _TEXT("%s\\%s\\%s\\%s\\%s"), BASE_PATH_CFG, BASE_NAME_CFG, inf->in_group_name.text, DEFAULT_CONNECT, LONG_NAMES_KEY);
    ctx->long_names = get_registry_long_names_flag(reg_path);
    ctx->file_name = NULL;
    ctx->folder = NULL;
    ctx->flags = 0;
#ifdef UNIX
    _tcscpy(ctx->lock_name, "fat12_lock");
    ctx->handle = NULL;
    if (open_named_mutex(ctx->lock_name, &ctx->lock )) {
	err = (DWORD)SCARD_E_READER_UNAVAILABLE;
	goto done;
    }
    ctx->locked = FALSE;
#else /* UNIX */
    ctx->handle = INVALID_HANDLE_VALUE;
    ctx->access = 0;
#endif /* UNIX */
    inf->context = ctx;
    ctx = NULL;
done:
    free_FOLDER_ITEM(items, items_number, full_buffer);
    if (ctx) {
	free(ctx->path);
	free(ctx->volume_label);
	free(ctx);
    }
    free(reg_path);
    return err;
}

/*!
 * \ingroup fat12_info_general
 * \brief    .
 * \return  
 */
static DWORD dir_context_dup(TSupSysContext *context, TSupSysInfo *info)
{
    TSupSysInfoContextDup *inf = (TSupSysInfoContextDup*)info;
    TFat12Context *dest;
    TFat12Context *src = (TFat12Context*)context;

    /*   . */
    SUPSYS_PRE_CONTEXT(context, TFat12Context);
    SUPSYS_PRE_INFO(info, TSupSysInfoContextDup);

    dest = calloc(1, sizeof(TFat12Context));
    if (dest == NULL)
	return (DWORD)NTE_NO_MEMORY;
    dest->path = NULL;
    if (src->path) {
	dest->path = malloc((_tcslen(src->path) + 1) * sizeof(TCHAR));
	if (!dest->path) {
	    free(dest);
	    return (DWORD)NTE_NO_MEMORY;
	}
	_tcscpy(dest->path, src->path);
    }
    dest->volume_label = NULL;
    if (src->volume_label) {
	dest->volume_label = malloc((_tcslen(src->volume_label) + 1) * sizeof(TCHAR));
	if (!dest->volume_label) {
	    free(dest);
	    return (DWORD)NTE_NO_MEMORY;
	}
	_tcscpy(dest->volume_label, src->volume_label);
    }
    dest->path_to_item = NULL;
    if (src->path_to_item) {
	dest->path_to_item = malloc((_tcslen(src->path_to_item) + 1) * sizeof(TCHAR));
	if (!dest->path_to_item) {
	    free(dest->path);
	    free(dest);
	    return (DWORD)NTE_NO_MEMORY;
	}
	_tcscpy(dest->path_to_item, src->path_to_item);
    }
    dest->long_names = src->long_names;
    dest->file_name = NULL;
    dest->folder = NULL;
    dest->flags = src->flags;
    _tcscpy(dest->nickname, src->nickname);
    _tcscpy(dest->connect, src->connect);
    dest->must_check_connect = get_check_mode_from_registry() == 1;
#ifdef UNIX
    dest->volume_serial = (unsigned long)-1;
    dest->machine = 0;
    dest->handle = NULL;
    dest->locked = FALSE;
    _tcscpy(dest->lock_name, src->lock_name);
    const DWORD err = open_named_mutex(dest->lock_name, &dest->lock);
    if (err) {
	free(dest->volume_label);
	free(dest->path_to_item);
	free(dest->path);
	free(dest);
	return err;
    }
#else /* UNIX */
    dest->access = 0;
    dest->handle = INVALID_HANDLE_VALUE;
#endif /* UNIX */
    inf->dest_context = dest;
    return ERROR_SUCCESS;
}


DWORD dir_unregister(TSupSysContext *context, TSupSysInfo *info)
{
    TFat12Context *ctx = (TFat12Context*)context;
    UNUSED(info);

    if (ctx) {
#ifdef _WIN32
	LocalFree(ctx->pSD);
#endif //_WIN32
	free(ctx->path);
	free(ctx->path_to_item);
#ifdef UNIX
	delete_mutex(ctx->lock);
#endif //UNIX
	free(ctx->volume_label);
	free(ctx);
    }
    return ERROR_SUCCESS;
}

/*++++
 *    .
 ++++*/
DWORD dir_unique_get(TSupSysContext *context, TSupSysInfo *info)
{
    TSupSysInfoText *inf = (TSupSysInfoText*)info;
    TFat12Context *ctx = (TFat12Context*)context;

    SUPSYS_PRE_CONTEXT( context, TFat12Context );
    SUPSYS_PRE_INFO( info, TSupSysInfoText );

    if (inf->text == NULL || inf->length == 0) {
	inf->length = _tcslen(ctx->volume_label);
	return ERROR_SUCCESS;
    }
    if (inf->length < _tcslen(ctx->volume_label)) {
	inf->length = _tcslen(ctx->volume_label);
	return ERROR_MORE_DATA;
    }
    _tcscpy(inf->text, ctx->volume_label);
    inf->length = _tcslen(ctx->volume_label);
    return ERROR_SUCCESS;
}

/*!
 * \ingroup fat12_internal
 * \brief  .
 */
static const unsigned char DIR_FLAGS_GENERAL[] = 
{ 
    0 << 7 
};

static const unsigned char DIR_FLAGS_CONNECT[] =
{
      0 << 7 /* SUPSYS_FLAG_CONNECT_MANY */
    | 0 << 6 /* SUPSYS_FLAG_CONNECT_COM */
    | 0 << 5 /* SUPSYS_FLAG_CONNECT_LPT */
    | 0 << 4 /* SUPSYS_FLAG_CONNECT_AUTO */
    | 0 << 3 /* SUPSYS_FLAG_CONNECT_MULTY */
    | 0 << 2 /* SUPSYS_FLAG_CONNECT_PNPGROUP */
};

static const unsigned char DIR_FLAGS_RDR[] =
{
    TO_CP_FLAG(RDR_FLAG_RDR_REMOVABLE, 1) |
    TO_CP_FLAG(RDR_FLAG_RDR_UNIQUE, 1) |
    TO_CP_FLAG(RDR_FLAG_RDR_PROTECTED, 0) |
    TO_CP_FLAG(RDR_FLAG_RDR_VIRTUAL, 0)
};

static const unsigned char DIR_FLAGS_GRP_CONNECT[] =
{
      0 << 7 /* SUPSYS_FLAG_CONNECT_MANY */
    | 0 << 6 /* SUPSYS_FLAG_CONNECT_COM */
    | 0 << 5 /* SUPSYS_FLAG_CONNECT_LPT */
    | 0 << 4 /* SUPSYS_FLAG_CONNECT_AUTO */
    | 0 << 3 /* SUPSYS_FLAG_CONNECT_MULTY */
    | 1 << 2 /* SUPSYS_FLAG_CONNECT_PNPGROUP */
};

/*++++
 *    .
 ++++*/
DWORD dir_info_system_flag(
    TSupSysContext *context, 
    TSupSysInfo *info ) /*+ (io)  . +*/
/*+
 * :
 *
 * ERROR_SUCCESS,     .
 *
 * (DWORD)ERROR_INVALID_PARAMETER,    .
 +*/
{
    TSupSysInfoFlags *flags = (TSupSysInfoFlags*)info;
    size_t length;
    const unsigned char *ptr = NULL;

    UNUSED( context );
    SUPSYS_PRE_INFO( info, TSupSysInfoFlags );
    SUPSYS_PRE( flags->type == RDR_FLAG_BLOCK_RDR
      || flags->type == SUPSYSE_FLAG_BLOCK_GENERAL
      || flags->type == SUPSYS_FLAG_BLOCK_CONNECT );
    length = flags->string.bits;
    if( flags->type == SUPSYSE_FLAG_BLOCK_GENERAL )
    {
	length = SUPSYSE_FLAG_GENERAL_VERSION_1;
	ptr = DIR_FLAGS_GENERAL;
    }
    if( flags->type == SUPSYS_FLAG_BLOCK_CONNECT )
    {
	length = SUPSYS_FLAG_CONNECT_VERSION_3;
	ptr = DIR_FLAGS_CONNECT;
    }
    if( flags->type == RDR_FLAG_BLOCK_RDR )
    {
	length = RDR_FLAG_RDR_VERSION_6;
	ptr = DIR_FLAGS_RDR;
    }
    if( flags->string.string == NULL )
    {
	flags->string.bits = length;
	return ERROR_SUCCESS;
    }
    if( flags->string.bits > length )
	flags->string.bits = length;
    memcpy( flags->string.string, ptr, 
	flags->string.bits / 8 + ( (flags->string.bits % 8) ? 1 : 0 ) );
    flags->string.bits = length;
    return ERROR_SUCCESS;
}

static DWORD get_flags_by_type(size_t type, const BYTE** bits, size_t* bits_len) {
    switch (type) {
    case SUPSYSE_FLAG_BLOCK_GENERAL:
	*bits_len = SUPSYSE_FLAG_GENERAL_VERSION_1;
	*bits = DIR_FLAGS_GENERAL;
	break;
    case SUPSYS_FLAG_BLOCK_CONNECT:
	*bits_len = SUPSYS_FLAG_CONNECT_VERSION_3;
	*bits = DIR_FLAGS_GRP_CONNECT;
	break;
    case RDR_FLAG_BLOCK_RDR:
	*bits_len = RDR_FLAG_RDR_VERSION_6;
	*bits = DIR_FLAGS_RDR;
	break;
    default:
	return (DWORD)ERROR_INVALID_PARAMETER;
    }

    return ERROR_SUCCESS;
}

DWORD dir_group_info_system_flag(
				     TSupSysContext *context, 
				     TSupSysInfo *info )
{
    TSupSysInfoFlags *flags = (TSupSysInfoFlags*)info;
    size_t length;
    const unsigned char *ptr = NULL;

    UNUSED( context );
    SUPSYS_PRE_INFO( info, TSupSysInfoFlags );
    DWORD err = get_flags_by_type(flags->type, &ptr, &length);
    if (err)
	return err;
    if( flags->string.string == NULL )
    {
	flags->string.bits = length;
	return ERROR_SUCCESS;
    }
    if( flags->string.bits > length )
	flags->string.bits = length;
    memcpy( flags->string.string, ptr, 
	flags->string.bits / 8 + ( (flags->string.bits % 8) ? 1 : 0 ) );
    flags->string.bits = length;
    return ERROR_SUCCESS;
}

static SUP_INLINE DWORD get_multi_folders(TCHAR * multifolders, size_t * size_multifolders)
{
    static const TCHAR ItemsPath[] = _TEXT("\\config\\KeyDevices\\DIR\\PNP DIR\\Default\\Items");
    DWORD err;
    size_t real_length = 0;
    err = support_registry_get_multi_string(ItemsPath, &real_length, NULL);
    if (err)
	goto done;
    if (real_length == 0) {
	err = (DWORD)ERROR_NO_MORE_ITEMS;
	goto done;
    }
    if (multifolders == NULL) {
	*size_multifolders = real_length;
	goto done;
    }
    if (real_length > *size_multifolders) {
	*size_multifolders = real_length;
	err = (DWORD)ERROR_MORE_DATA;
	goto done;
    }
    err = support_registry_get_multi_string(ItemsPath, size_multifolders, multifolders);
done:
    return err;
}

static DWORD multistring_count_strings(const TCHAR* multistring_buf) 
{
    DWORD res = 0;
    const TCHAR* cur_pointer = multistring_buf;
    while (*cur_pointer != _TEXT('\0')) {
	cur_pointer = cur_pointer + _tcslen(cur_pointer) + 1;
	res++;
    }
    return res;
}

static TCHAR * delete_spaces(TCHAR * src)
{
    TCHAR * pos = src;

    if (src == NULL)
	return NULL;
    while (pos[0] != _TEXT('\0')) {
	if (pos[0] == _TEXT(' ') || pos[0] == _TEXT('\n') || pos[0] == _TEXT(';') || pos[0] == _TEXT('?') || pos[0] == _TEXT('*'))
	    pos++;
	else break;
    }
    while (pos[0] != _TEXT('\0')) {
	TCHAR * last_pos = pos + _tcslen(pos) - 1;
	if (last_pos[0] == _TEXT(' ') || last_pos[0] == _TEXT('\n') ||
	    last_pos[0] == _TEXT(';') || last_pos[0] == _TEXT('?') ||
	    last_pos[0] == _TEXT('*'))
	    last_pos[0] = _TEXT('\0');
	else break;
    }
    if (pos[0] == _TEXT('\0'))
	return NULL;
    return pos;
}

static DWORD get_clear_path(TCHAR * meta_path, TCHAR ** clear_path)
{
    TCHAR * path_wo_spaces = delete_spaces(meta_path);
#ifdef _WIN32
    TCHAR * ret = NULL;
    DWORD size = 0; 
    size = ExpandEnvironmentStrings(path_wo_spaces, NULL, 0);
    if (!size)
	return (DWORD)ERROR_INVALID_PARAMETER;
    ret = malloc(size * sizeof(TCHAR));
    if (!ret)
	return (DWORD)NTE_NO_MEMORY;
    size = ExpandEnvironmentStrings(path_wo_spaces, ret, size);
    if (!size) {
	free(ret);
	return (DWORD)ERROR_INVALID_PARAMETER;
    }
    *clear_path = ret;
#else /*_WIN32*/
    *clear_path = path_wo_spaces;
#endif /*_WIN32*/
    return ERROR_SUCCESS;
}

DWORD get_items_from_registry(TSupSysComputeCheckMark compute_mark_func, unsigned short * change_mark, 
    FOLDER_ITEM ** out_items, DWORD * out_items_number, TCHAR ** full_buffer, size_t * max_name_length)
{
    DWORD err;
    size_t mf_size = 0;
    TCHAR * multifolders = NULL;
    DWORD items_number = 0, iter;
    FOLDER_ITEM * items = NULL;
    TCHAR * pos = NULL;
    size_t pos_len;
    size_t all_len = 0;
    size_t max_length = 0;

    err = get_multi_folders(NULL, &mf_size);
    if (err)
	goto done;
    multifolders = malloc((mf_size + 1) * sizeof(TCHAR));
    if (!multifolders) {
	err = (DWORD)NTE_NO_MEMORY;
	goto done;
    }
    err = get_multi_folders(multifolders, &mf_size);
    if (err)
	goto done;
    if (compute_mark_func)
#ifdef _WIN32
	compute_mark_func(multifolders, change_mark);
#else //_WIN32
	*change_mark = 0;
#endif //_WIN32
    items_number = multistring_count_strings(multifolders);
    if (items_number == 0) {
	err = (DWORD)ERROR_NO_MORE_ITEMS;
	goto done;
    }
    items = calloc(items_number, sizeof(FOLDER_ITEM));
    if (!items) {
	err = (DWORD)NTE_NO_MEMORY;
	goto done;
    }
    iter = 0;
    pos = multifolders;
    do {
	TCHAR * left_bracket = _tcschr(pos, _TEXT('('));
	TCHAR * right_bracket = _tcschr(pos, _TEXT(')'));
	TCHAR * eq = _tcschr(pos, _TEXT('='));
	pos_len = _tcslen(pos) + 1;
	all_len += pos_len;
	if (right_bracket == NULL || left_bracket == NULL || eq == NULL) {
	    err = (DWORD)ERROR_NO_MORE_ITEMS;
	    goto done;
	}
	if (right_bracket < left_bracket || eq < right_bracket) {
	    err = (DWORD)ERROR_NO_MORE_ITEMS;
	    goto done;
	}
	left_bracket[0] = _TEXT('\0');
	right_bracket[0] = _TEXT('\0');
	eq[0] = _TEXT('\0');

	items[iter].name = delete_spaces(pos);
	items[iter].unique = delete_spaces(left_bracket + 1);
	err = get_clear_path(delete_spaces(eq + 1), &items[iter].path);
	if (err) 
	    goto done;
	if (items[iter].name == NULL || items[iter].unique == NULL || items[iter].path == NULL) {
	    err = (DWORD)ERROR_NO_MORE_ITEMS;
	    goto done;
	}
	if (max_length < _tcslen(items[iter].name))
	    max_length = _tcslen(items[iter].name);
	pos += pos_len;
	iter++;
    } while (pos[0] != _TEXT('\0') && all_len < mf_size && iter < items_number);
    *out_items = items;
    items = NULL;
    *out_items_number = items_number;
    *full_buffer = multifolders;
    multifolders = NULL;
    *max_name_length = max_length/* + _tcslen(_TEXT("DIR_"))*/;
done:
    free_FOLDER_ITEM(items, items_number, multifolders);
    return err;
}


/*! 
* \ingroup fat12_fun_reader
* \brief       .
*
*      .
* \param context [in]  . 
* \param info [in/out]  #TReaderInfoConnectOpen
* \sa #TReaderInfoConnectOpen, #READER_FUN_CONNECT_NEXT, 
*  #TSupSysFunction, #fat12_info_connect_close, #fat12_info_connect_next,
* \return  
* \retval #ERROR_SUCCESS    
*  
* \retval #(DWORD)ERROR_INVALID_PARAMETER   
* \retval #(DWORD)NTE_NO_MEMORY   
* \retval " " \ref fat12_os_error. 
*/
DWORD dir_group_enum_open(
    TSupSysContext *context, 
    TSupSysInfo *info )
{
    TSupSysInfoGroupEnum *inf = (TSupSysInfoGroupEnum*)info;
    TDirConnectEnum *enum_ptr;
    DWORD err;

    UNUSED( context );
    SUPSYS_PRE_INFO(info, TSupSysInfoGroupEnum);

    inf->add_info.length = 0;
    inf->add_info.info = NULL;
    
    enum_ptr = calloc(1, sizeof(TDirConnectEnum));
    if (enum_ptr == NULL)
	return (DWORD)NTE_NO_MEMORY;
    err = get_items_from_registry(inf->compute_mark_func, &inf->change_mark, &enum_ptr->items, &enum_ptr->number, &enum_ptr->full_buffer, &inf->name.length);
    if (err)
	goto done;
    enum_ptr->cur = 0;
    inf->add_info.length = sizeof(TDirConnectEnum);
    inf->add_info.info = (unsigned char*)enum_ptr;
    enum_ptr = NULL;
done:
    free(enum_ptr);
    return err;
}

/*++++
*     .
++++*/
DWORD dir_group_enum_next(
    TSupSysContext *context, 
    TSupSysInfo *info )
{
    TSupSysInfoGroupEnum *inf = (TSupSysInfoGroupEnum*)info;
    TDirConnectEnum *enum_ptr;

    UNUSED(context);
    SUPSYS_PRE_INFO(info, TSupSysInfoGroupEnum);
    SUPSYS_PRE_READ_PTRS(inf->add_info.info, sizeof(TDirConnectEnum));

    enum_ptr = (TDirConnectEnum*)(inf->add_info.info);
    if (enum_ptr->cur < enum_ptr->number) {
	if (_tcslen(enum_ptr->items[enum_ptr->cur].name) > inf->name.length)
	    return (DWORD)ERROR_INVALID_PARAMETER;
	_tcscpy(inf->name.text, enum_ptr->items[enum_ptr->cur].name);
	enum_ptr->cur++;
	return ERROR_SUCCESS;
    } else 
	return (DWORD)SCARD_E_READER_UNAVAILABLE;
}

DWORD dir_group_enum_close(
    TSupSysContext *context, 
    TSupSysInfo *info )
{
    TSupSysInfoGroupEnum *inf = (TSupSysInfoGroupEnum*)info;
    TDirConnectEnum *enum_ptr;

    UNUSED( context );
    SUPSYS_PRE_INFO( info, TSupSysInfoGroupEnum);

    enum_ptr = (TDirConnectEnum*)inf->add_info.info;
    if( enum_ptr == NULL )
    {
	inf->add_info.length = 0;
	return ERROR_SUCCESS;
    }
    free_FOLDER_ITEM(enum_ptr->items, enum_ptr->number, enum_ptr->full_buffer);
    free(enum_ptr);
    inf->add_info.length = 0;
    inf->add_info.info = NULL;
    return ERROR_SUCCESS;
}

static DWORD check_dir_is_connected(TFat12Context *ctx)
{
    DWORD err = ERROR_SUCCESS;
    if (ctx->must_check_connect) {
	FOLDER_ITEM * items = NULL;
	DWORD items_number = 0, i = 0;
	TCHAR * full_buffer = NULL;
	size_t max_name_length = 0;
	err = get_items_from_registry(NULL, NULL, &items, &items_number, &full_buffer, &max_name_length);
	if (err)
	    goto done;

	for (i = 0; i < items_number; i++) {
	    if (!_tcscmp(items[i].name, ctx->connect) && !_tcscmp(items[i].unique, ctx->volume_label))
		goto done;
	}
	err = (DWORD)SCARD_E_NO_SMARTCARD;
    }
done:
    return err;
}

//,    p_call_ctx
static DWORD dir_lock(TSupSysContext* context, TSupSysInfo* info) {
    TFat12Context *ctx = (TFat12Context*)context;
    UNUSED(info);
    return check_dir_is_connected(ctx);
}

//   
static DWORD dir_unlock(TSupSysContext* context, TSupSysInfo* info) {
    UNUSED(context);
    UNUSED(info);
    return ERROR_SUCCESS;
}

/*+   info-. +*/
static const TSupSysFunctionTableItem dir_funs[] =
{
    { SUPSYS_FUN_NICKNAME, fat12_info_nickname },
    { SUPSYS_FUN_NAME, dir_info_name },
    { SUPSYS_FUN_INSTANCE, fat12_info_instance },
    { SUPSYS_FUN_CONTEXT_DUP, dir_context_dup },
    { SUPSYS_FUN_CONTEXT_FREE, fat12_context_free },
    { SUPSYS_FUN_FLAGS, dir_info_system_flag },
#ifndef UNIX
    { SUPSYS_FUN_CALL, fat12_aux_call },
#endif
    { SUPSYS_FUN_VERSIONSUPPORT, info_versionsupport},
    { SUPSYS_FUN_REGISTER, dir_register },
    { SUPSYS_FUN_UNREGISTER, dir_unregister },

    { READER_FUN_LOCK, dir_lock },
    { READER_FUN_UNLOCK, dir_unlock },

    { READER_FUN_CONNECT_CARRIER, fat12_connect_carrier },
    { READER_FUN_DISCONNECT_CARRIER, fat12_disconnect_carrier },
    
    { READER_FUN_GET_PARAM, fat12_get_param },
    { READER_FUN_PASSWORD_TYPE, fat12_get_password_type },
    { READER_FUN_GET_APPLET_TEXT_INFO, hdimage_get_text_info },

    { READER_FUN_UNIQUE_GET, dir_unique_get },

    { READER_FUN_FOLDER_ENUM_OPEN, fat12_info_folder_enum_open },
    { READER_FUN_FOLDER_ENUM_NEXT, fat12_info_folder_enum_next },
    { READER_FUN_FOLDER_ENUM_CLOSE, fat12_info_folder_enum_close },

    { READER_FUN_FOLDER_OPEN, fat12_folder_open },
    { READER_FUN_FOLDER_CLOSE, fat12_folder_close },

#ifdef _WIN32
    { READER_FUN_SEC_DESCR_READ, fat12_folder_get_sdecr },
#endif //_WIN32
    
    { READER_FUN_FOLDER_CLEAR, fat12_folder_clear },
    { READER_FUN_OPEN, fat12_open },
    { READER_FUN_CLOSE, fat12_close },
    { READER_FUN_UNLINK, fat12_unlink },

#ifdef _WIN32
    { READER_FUN_SEC_DESCR_WRITE, fat12_folder_put_sdecr },
#endif //_WIN32

    { READER_FUN_READ, fat12_read },
    { READER_FUN_WRITE, fat12_write },
    { READER_FUN_CHSIZE, fat12_chsize },
    { READER_FUN_LENGTH, fat12_length },
    { READER_FUN_GET_FREE_SPACE, fat12_free_space },
};

static const TSupSysFunctionTableItem DIR_GROUP_FUNS[] =
{
    { SUPSYS_FUN_NICKNAME, dir_info_group_nickname },
    { SUPSYS_FUN_NAME, dir_info_group_name },
    { SUPSYS_FUN_INSTANCE, fat12_info_instance },
    { SUPSYS_FUN_FLAGS, dir_group_info_system_flag },
    { SUPSYS_FUN_GROUP_ENUM_OPEN, dir_group_enum_open },
    { SUPSYS_FUN_GROUP_ENUM_NEXT, dir_group_enum_next },
    { SUPSYS_FUN_GROUP_ENUM_CLOSE, dir_group_enum_close },
    { SUPSYS_FUN_GROUP_NICKNAME, dir_info_group_group_nickname },
    { SUPSYS_FUN_VERSIONSUPPORT, info_versionsupport},
};

static const TSupSysFunctionTable DIR_GROUP_TABLE =
{
    sizeof( DIR_GROUP_FUNS ) / sizeof( TSupSysFunctionTableItem ),
    DIR_GROUP_FUNS
};

static const TSupSysFunctionTable dir_table =
{
    sizeof( dir_funs ) / sizeof( TSupSysFunctionTableItem ),
    dir_funs
};

CPRO_PUBLIC_API const TSupSysEHandle *dir_reader_get_table(void)
{
    return (const TSupSysEHandle*)&dir_table;
}

CPRO_PUBLIC_API
const TSupSysEHandle *
dir_reader_get_group_table(void)
{
    return (const TSupSysEHandle*)&DIR_GROUP_TABLE;
}

