#ifndef _IKEV2G_PRF_H_
#define _IKEV2G_PRF_H_

typedef struct gost_prf_t gost_prf_t;

#include "blob.h"
#include "types.h"

/**
  * @brief HMAC interface.
  *
  * Can accumulate seed from keys or bytes.
  * HMAC key can be passed in constructor as key handle
  * or set as bytes via @ref set_key.
  */
struct gost_prf_t {

    /**
     * @brief Get hash value.
     *
     * Saves hash value into @p buffer and reopens hash.
     *
     * @param this	Pointer to the gost_prf_t structure.
     * @param buffer	Output buffer where hash value will be stored.
     * @return		True on success.
     */
    bool (*get_bytes)(gost_prf_t *this, uint8_t *buffer)
	WARN_UNUSED_RESULT;

    /**
     * @brief Get hash value.
     *
     * Allocates blob, saves hash value into @p blob and reopens hash.
     *
     * @param this	Pointer to the gost_prf_t structure.
     * @param blob	Uninitialized or empty blob where hash will be stored.
     * @return		True on success.
     */
    bool (*allocate_bytes)(gost_prf_t *this, blob_t *blob)
	WARN_UNUSED_RESULT;

    /**
     * @brief Derive key.
     *
     * Key length depends on prf block size.
     *
     * @param this	Pointer to the gost_prf_t structure.
     * @return		Derived key handler on success or 0 otherwise.
     */
    uintptr_t (*derive_key)(gost_prf_t *this);

    /**
     * @brief Add more data to the hash.
     *
     * Adds more seed data to the hash.
     *
     * @param this	Pointer to the gost_prf_t structure.
     * @param seed	Seed data.
     * @return		True on success.
     */
    bool (*seed_bytes)(gost_prf_t *this, blob_t seed);

    /**
     * @brief Add more key data to the hash.
     *
     * @param this	Pointer to the gost_prf_t structure.
     * @param key	Key handler.
     * @return		True on success.
     */
    bool (*seed_key)(gost_prf_t *this, uintptr_t key);

    /**
     * @brief Set HMAC key.
     *
     * @param this	Pointer to the gost_prf_t structure.
     * @param blob	Key blob.
     * @return		True on success.
     */
    bool (*set_key)(gost_prf_t *this, blob_t blob);

    /**
     * @brief Destructor of gost_prf_t.
     *
     * @param this	Pointer to the ikev2_gost_t structure.
     */
    void (*destroy)(gost_prf_t *this);
};

/**
 * @brief Create PRF from HMAC.
 *
 * Key for HMAC could be setted later with @ref set_key method.
 *
 * @param prov		CSP context.
 * @param hmac		HMAC algorithm id.
 * @param key		Key handle (optional).
 */
gost_prf_t *gost_prf_create(uintptr_t prov, IKE_GOST_HMAC_ID hmac,
    uintptr_t key);

#endif /* _IKEV2G_PRF_H_ */

